// =============================================================
// UART with 16-bit Baud Rate Register
//
// Author   : Klaus Kohl-Schoepe
// Date     : 11.05.2020
// File Name: myUART.v
// Copyright (C) 2020 Klaus Kohl-Schoepe (kks@designin.de)
// =============================================================

// =============================================================
// Verilog UART with 16-bit Baud Rate Register (CLK / Baudrate - starting 2400 Baud @ 100MHz)
// It will use 8 bit without parity and 1 stop bit (receiving: 0.5 stop bit)
// Actual without check of error on data line - only overflow during read and write
//
// Short description:
//  rx_data and tx_data are the 8-bit data interface
//  rx_read and tx_write are the read/write signal - should be set for read/write
//  rx_read_ack and rx_write_ack signalize that data are handled and rx_read or tx_write can be reset for next time
//
//  all other signals are status informations:
//    rx_full, tx_full         - new data are available or output buffer full
//    rx_overflow, tx_overflow - to many new rx data or double write to tx buffer
//    rx_error, tx_error       - reserved for error but actual not used
// =============================================================

//`timescale 100 ns / 10 ns
`timescale 1 ns / 1 ps

// =============================================================
// Modul myUART
// =============================================================

module myUART
(
  input  clk,               // The master clock
  input  [15:0] baudrate,   // Baudrate divider (speed = clk/baudrate)
  input  reset,             // Synchronous reset
  input  rx,                // RX line
  output reg tx,            // TX line
  // RX
  output rx_active,         // RX active.
  output reg rx_error,      // RX receive error
  output reg rx_overflow,   // RX not read before next byte (overwritten)
  output reg rx_full,       // A byte received
  input rx_read,            // RX byte have read
  output reg rx_read_ff,    // wait for negative flag for next read
  output reg [7:0] rx_data, // Received data
  // TX
  output tx_active,         // Low when transmit line is idle
  output reg tx_error,      // TX transmit error (not used)
  output reg tx_overflow,   // TX overwritten
  output reg tx_full,       // TX output buffer full
  input  tx_write,          // Assert to begin transmission
  output reg tx_write_ff,   // wait for negative flag for next write
  input [7:0] tx_data       // Byte to transmit
);

  // Working register like baud rate and status
  localparam [1:0]
    X_IDLE   = 2'd0, // no operation / wait for start bit
    X_START  = 2'd1, // wait one/skip half start bit
    X_BITS   = 2'd2, // wait for next read/write
    X_STOP   = 2'd3; // wait one bit at stop

  // for RX
  reg [ 2:0] rx_state = X_IDLE; // RX state
  reg [15:0] rx_bcount;         // Baud rate down counter for TX
  reg [ 3:0] rx_bits;           // Received bits
  reg [ 7:0] rx_dataw;          // RX data (working register)
   
  // for TX
  reg [ 2:0] tx_state = X_IDLE; // TX state
  reg [15:0] tx_bcount;         // Baud rate down counter for RX
  reg [ 3:0] tx_bits;           // Transmitted bits
  reg [ 7:0] tx_dataw;          // TX data (working register)
  reg [ 7:0] tx_datab;          // TX data puffer (next character)

  // Assignment
  assign rx_active = (rx_state != X_IDLE);
  assign tx_active = (tx_state != X_IDLE);
 
  // Initialization
  initial begin
    rx_state    = X_IDLE;
    rx_error    = 1'b0;
    rx_overflow = 1'b0;
    rx_full     = 1'b0;
    rx_read_ff  = 1'b0;
    rx_data     = 8'b0;
        
    tx          = 1'b1;
    tx_state    = X_IDLE;
    tx_error    = 1'b0;
    tx_overflow = 1'b0;
    tx_full     = 1'b0;
    tx_write_ff = 1'b0;
  end

// =============================================================
// Main Loop
// =============================================================
  // Check reset
  always @(posedge clk) begin
    if (reset) begin
      rx_state    = X_IDLE;
      rx_error    = 1'b0;
      rx_overflow = 1'b0;
      rx_full     = 1'b0;
      rx_data     = 8'b0;
      rx_read_ff  = 1'b0;
        
      tx          = 1'b1;
      tx_state    = X_IDLE;
      tx_error    = 1'b0;
      tx_overflow = 1'b0;
      tx_full     = 1'b0;
      tx_write_ff = 1'b0;
    end

    // Check if data has read
    if (rx_read) begin
      if (!rx_read_ff) begin
        rx_error    = 1'b0;
        rx_overflow = 1'b0;
        rx_full     = 1'b0;
        rx_read_ff  = 1'b1;
      end
    end else begin
        rx_read_ff = 1'b0;
    end

    // Decrement receiver counter
    if(rx_bcount) begin
        rx_bcount = rx_bcount - 1'd1;
    end

    // Begin: Receiver state machine
    case (rx_state)
      X_IDLE: begin
        if (!rx) begin          // start if RX low
            rx_state = X_START; // wait 1/2 start bit
            rx_bcount = { 1'b0, baudrate[15:1] };
        end
      end
      
      X_START: begin
        if (!rx_bcount) begin   // after 1/2 bit
          rx_state = X_BITS;    // wait for first data bit
          rx_bcount = baudrate;
          rx_bits = 4'd8;       // read 8 bits
        end
      end
        
      X_BITS: begin
        if (!rx_bcount) begin
          rx_bcount = baudrate;           // wait again one bit for next date / Stop
          rx_dataw = {rx, rx_dataw[7:1]}; // read MSB and add to rest of data
          rx_bits = rx_bits - 1'd1;       // bit count - 1
          if (!rx_bits) begin             // all 8 bits ?
            rx_state = X_STOP;            // waiting for mid of stop bit
          end
        end
      end

      X_STOP: begin
        if (!rx_bcount) begin
          if (!rx_bits) begin     // Still stop bit ?
            rx_state = X_IDLE;    // Yes: data received, wait for start
            rx_data = rx_dataw;   // make data available
            if(rx_full) begin     // if data still in buffer
              rx_overflow = 1'b1; // then set overflow flag
            end
            rx_full = 1'b1;
          end else begin          // Stop error:
            rx_error = 1'b1;      // then set error flag
          end
        end
      end            
    endcase
    // End: Receiver state machine
        
    // Check if data available for TX
    if (tx_write) begin           // If request for write
      if (!tx_write_ff) begin     // if no acknowlede
        tx_datab = tx_data;       // get data to buffer
        if (tx_full) begin        // old data overwritten ?
          tx_overflow = 1'b1;     // yes: set overflow flag
        end else begin
          tx_overflow = 1'b0;     // no: reset overflow flag
          tx_error = 1'b0;        //           and error flag
        end
        tx_full = 1'b1;           // Output buffer now full
        tx_write_ff = 1'b1;       // send acknowledge
      end
    end else begin                // If no write
      tx_write_ff = 1'b0;         // reset acknowledge
    end

    // Transmitter counter
    if(tx_bcount) begin
        tx_bcount = tx_bcount - 1'd1;
    end

    // Begin: Transmitter state machine
    case (tx_state)
      X_IDLE: begin
        if (tx_full) begin        // If data available
          tx_state = X_BITS;      // wait for first bit to send
          tx_bcount = baudrate;
          tx = 1'b0;              // Start bit is 0
          tx_dataw = tx_datab;    // data from buffer
          tx_bits = 4'd8;         // write 8 bits
          tx_full = 1'b0;         // TX buffer now free
        end
      end
    
      X_START: begin              // Start bit included above
      end
          
      X_BITS: begin
        if (!tx_bcount) begin
          if (!tx_bits) begin     // all 8 bits ?
            tx_bcount = baudrate; // wait one bit for stop
            tx_state = X_STOP;    // waiting for end of stop bit
            tx = 1'b1;            // Sendig stop bit
          end else begin
            tx_bcount = baudrate; // wait again one bit for next date / Stop
            tx = tx_dataw[0];     // sending LSB
            tx_dataw = {1'b0, tx_dataw[7:1]}; // shift rest of data right
            tx_bits = tx_bits - 1'd1;  // count - 1
          end
        end
      end

      X_STOP: begin
        if (!tx_bcount) begin
          tx_state = X_IDLE;      // data send
        end
      end            
    endcase
    // End: Transmitter state machine
  end

endmodule
